/**
 * \file: svg_bitmap_decoder.h
 *
 * \version: $Id: svg_bitmap_decoder.h,v 1.17 2010/01/14 13:43:29 tkniep Exp $ 
 *
 * This header file declares the functions and types that are provided by the SVG bitmap decoder
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/
 
#ifndef SVG_BITMAP_DECODER_H
#define SVG_BITMAP_DECODER_H

/* Information for automatic subsystem stub generation */
/*** DEFINE_IFLIB

[LIBRARY HEADER FILE]
ifsvg_bitmap_decoder.h

[FNUMBER HEADER FILE]
fnsvg_bitmap_decoder.h

[INCLUDE FILE]
"svg_bitmap_decoder.h"

[PREFIX]
SVG_BMPDEC

***/

#ifdef __cplusplus
extern "C" {
#endif

#include <svg_typedef.h>
#include <svg_error.h>
#include <pthread.h>
#include <grl_os_abstraction.h>
#include "svg_bitmap_decoder_err.h"

/*******************************************************************************************
 *   Macro definitions
 *******************************************************************************************/

/** Module version */
#define SVG_BMPDEC_MODULE_VER           "$Name: SVG_r2_C01 $"

/** Bitmasks for indicating the supported decoders. See SVGBmpDecFormats */
#define SVG_BMPDEC_FORMAT_JPEG          (0x1 << (SVGUint32)SVG_BMPDEC_JPEG_ENCODING)    /**< JPEG decoder available     */
#define SVG_BMPDEC_FORMAT_PNG           (0x1 << (SVGUint32)SVG_BMPDEC_PNG_ENCODING)     /**< PNG decoder available      */
#define SVG_BMPDEC_FORMAT_FGS           (0x1 << (SVGUint32)SVG_BMPDEC_FGS_ENCODING)     /**< PNG decoder available      */


/** Two queues are supported. Do not change this value! */
#define GRL_BMPDEC_QU_NUM               (2)

/*******************************************************************************************
 *   Global variables
 *******************************************************************************************/

/* Global Version Information */
extern const SVGUint8 g_SVGBMPDEC_version[];

/*******************************************************************************************
 *   Public Type Definitions
 *******************************************************************************************/

/*****************************************
 *   FGS specific
 *****************************************/
typedef struct SVGBmpDecAttr_FGS_
{
    SVGBoolean          use_colorkey;           /* TRUE: use FALSE: don't use   */
} SVGBmpDecAttr_FGS;


/*****************************************
 *   JPEG specific
 *****************************************/

/**
 * Different rotation angles which are supported by the JPEG decoder
 */
typedef enum SVGBmpDecAttrRot_JPEG_
{
    SVG_BMPDEC_ROTATE_0 = 0,            /**< 0 degrees rotation for JPEG decoding       */
    SVG_BMPDEC_ROTATE_90,               /**< 90 degrees rotation for JPEG decoding      */
    SVG_BMPDEC_ROTATE_180,              /**< 180 degrees rotation for JPEG decoding     */
    SVG_BMPDEC_ROTATE_270               /**< 270 degrees rotation for JPEG decoding     */
} SVGBmpDecAttrRot_JPEG;


/**
 * Decoder parameters specific for JPEG
 */
typedef struct SVGBmpDecAttr_JPEG_
{
    SVGBoolean                  effect_enable;  /**< Enable or disable tone effects     */
    SVGUint8                    tone_u;         /**< Red element                        */
    SVGUint8                    tone_v;         /**< Blue element                       */
    SVGBmpDecAttrRot_JPEG       rotation;       /**< Rotation angle                     */
} SVGBmpDecAttr_JPEG;


/**
 * Image information attributes specific for JPEG
 */
typedef struct SVGBmpDecInfoJPEG_
{
    SVGUint8                    orientation;    /**< Orientation of image               */
    SVGUint8                    datetime[20];   /**< Date and time of image             */
} SVGBmpDecInfoJPEG;

/*****************************************
 *   PNG specific
 *****************************************/


/**
 * All color modes which are supported by PNG
 */
typedef enum SVGBmpDecPNGColorMode_
{
    SVG_BMPDEC_PNG_COLOR_TYPE_UNKNOWN = 0,      /**< Color mode is unknown or undefined */
    SVG_BMPDEC_PNG_COLOR_TYPE_GRAY,             /**< Grayscale image                    */
    SVG_BMPDEC_PNG_COLOR_TYPE_GRAY_ALPHA,       /**< Grayscale image with alphachannel  */
    SVG_BMPDEC_PNG_COLOR_TYPE_PALETTE,          /**< Palette (indexed) image            */
    SVG_BMPDEC_PNG_COLOR_TYPE_RGB,              /**< RGB image                          */
    SVG_BMPDEC_PNG_COLOR_TYPE_RGB_ALPHA         /**< RGB image with alphachannel        */
} SVGBmpDecPNGColorMode;


/**
 * All filter methods which are supported by PNG
 */
typedef enum SVGBmpDecPNGFilterMethod_
{
    SVG_BMPDEC_PNG_FILTER_METHOD_UNKNOWN = 0,   /**< Filter method is unknown or undefined      */
    SVG_BMPDEC_PNG_FILTER_METHOD_BASE           /**< Filter method 0, the only supported one    */
} SVGBmpDecPNGFilterMethod;


/**
 * Time information about when a PNG file was last modified
 */
typedef struct SVGBmpDecPNGTime_
{
    SVGUint16           year;                   /**< Full year                          */
    SVGUint8            month;                  /**< Month of the year                  */
    SVGUint8            day;                    /**< Day of the month                   */
    SVGUint8            hour;                   /**< Hour of the day                    */
    SVGUint8            minute;                 /**< Minute of the hour                 */
    SVGUint8            second;                 /**< Second of the minute               */
} SVGBmpDecPNGTime;


/**
 *
 */
typedef struct SVGBmpDecInfoPNG_
{
    SVGBmpDecPNGColorMode       color_mode;             /**< Color mode of PNG image            */
    SVGBoolean                  has_alphachannel;       /**< Image has alphachannel             */
    SVGBoolean                  is_interlaced;          /**< Image is interlaced                */
    SVGBmpDecPNGFilterMethod    filter_method;          /**< Filter method for compression      */
    SVGUint32                   has_bg_color;           /**< Image has background color         */
    SVGUint32                   bg_color;               /**< Background color                   */
    SVGBoolean                  has_mod_time;           /**< PNG has modification time info     */
    SVGBmpDecPNGTime            mod_time;               /**< Modification time                  */
} SVGBmpDecInfoPNG;

/**
 * Decoder parameters specific for PNG
 */
typedef struct SVGBmpDecAttr_PNG_
{
    SVGInt16            plte_chunk_size;        /**< Size of palette chunk in bytes             */
    SVGInt8             *plte_chunk;            /**< Pointer to palette chunk (PLTE)            */
    SVGInt16            trns_chunk_size;        /**< Size of transparency chunk in bytes        */
    SVGInt8             *trns_chunk;            /**< Pointer to transparency chunk (tRNS)       */
} SVGBmpDecAttr_PNG;


/*****************************************
 *   Common (all formats)
 *****************************************/

/** 
 * Clip rect dimensions
 */
typedef struct SVGBmpDecRect_
{
    SVGUint16            x;             /* Clipping position x  */
    SVGUint16            y;             /* Clipping position y  */
    SVGUint16            width;         /* Clipping size width  */
    SVGUint16            height;        /* Clipping size height */
} SVGBmpDecRect;


/**
 * All supported encoding formats
 */
typedef enum SVGBmpDecImageEncoding_
{
    SVG_BMPDEC_UNKNOWN_ENCODING = 0,    /**< Undefined encoding                           */
    SVG_BMPDEC_JPEG_ENCODING,           /**< JPEG compressed image                        */
    SVG_BMPDEC_PNG_ENCODING,            /**< PNG compressed image                         */
    SVG_BMPDEC_JPEG_THUMBNAIL,          /**< JPEG compressed thumbnail                    */
    SVG_BMPDEC_PNG_ENCODING_WITH_PLTE,  /**< PNG compressed image with external palette   */
    SVG_BMPDEC_FGS_ENCODING,			/**< FGS encoding                              */
    SVG_BMPDEC_ENCODING_MAX_VALUE
} SVGBmpDecImageEncoding;


/**
 * Supported decoding modes
 */
typedef enum SVGBmpDecMode_
{
    SVG_BMPDEC_MODE_UNKNOWN = 0,        /**< Undefined mode                               */
    SVG_BMPDEC_MODE_EGL_SURFACE,        /**< Decode to a EGL Surface (not supported)      */
    SVG_BMPDEC_MODE_MEMORY,             /**< Decode directly to memory                    */
    SVG_BMPDEC_MODE_MAX_VALUE
} SVGBmpDecMode;


/**
 * Contains a struct for each supported decoding module. Used for
 * passing encoding-specific information to the decoder
 */
typedef union SVGBmpDecAttr_
{
    SVGBmpDecAttr_JPEG  jpeg;                /**< JPEG-specific attributes                           */
    SVGBmpDecAttr_PNG   png;                 /**< PNG-specific attributes                            */
    SVGBmpDecAttr_FGS   fgs;                 /**< FGS-specific attributes                         */
} SVGBmpDecAttr;


/**
 * Memory decoding specific settings
 */
typedef struct SVGBmpDecDestMem_
{
    SVGUint8            *dest_mem;              /**< Pointer to destination memory                      */
    SVGUint32           dest_size;              /**< Size of destination memory                         */
} SVGBmpDecDestMem;


/**
 * Contains a struct for each supported decoding module. Used for
 * passing destination details to the decoder
 */
typedef union SVGBmpDecDestSettings_
{
    SVGBmpDecDestMem    dest_mem;               /**< Destination details for direct memory decoding     */
} SVGBmpDecDestSettings;


/**
 * Contains a struct for each supported encoding format. Used for retrieving encoding-specific
 * image information from the decoder.
 */
typedef union SVGBmpDecInfoFormat_
{
    SVGBmpDecInfoJPEG   jpeg;                   /**< JPEG-specific image information                    */
    SVGBmpDecInfoPNG    png;                    /**< PNG-specific image information                     */
} SVGBmpDecInfoFormat;


/**
 * Image definition structure               
 */
typedef struct SVGImage_
{
    SVGBmpDecImageEncoding      type;           /**< Encoding type.                                     */
    SVGBmpDecMode               decode_mode;    /**< Decode mode (e.g. SVG Surface )                    */
    SVGBmpDecDestSettings       dest;           /**< Destination for decoding                           */
    SVGUint16                   width;          /**< Image size (width)                                 */
    SVGUint16                   height;         /**< Image size (height)                                */
    SVGBmpDecRect               clip;           /**< Image clipping area of the source image            */
    SVGUint32                   color_key;      /**< Color key (in ARGB8888)                            */
    SVGUint32                   data_size;      /**< Data size of compressed image                      */
    SVGInt8                     *datap;         /**< Pointer to compressed image                        */
    SVGBmpDecAttr               attribute;      /**< Decoder-specific attributes                        */
} SVGImage;


/**
 * Image information structure
 */
typedef struct SVGImageInfo_
{
    SVGBmpDecImageEncoding      type;           /**< Encoding type                                      */
    SVGUint16                   width;          /**< Image size (width)                                 */
    SVGUint16                   height;         /**< Image size (height)                                */
    SVGInt32                    bit_depth;      /**< Bit depth of color                                 */
    SVGBmpDecInfoFormat         format;         /**< Decoder-specific image info                        */
} SVGImageInfo;


/**
 * Bitmask indicating the supported formats
 * Should be accessed only with combinations of the macros
 * SVG_BMPDEC_FORMAT_*
 */
typedef SVGUint32       SVGBmpDecFormats;
 

/*******************************************************************************************
 *   Internal Type Definitions (to be used only by the Decoder Core and Decoder  Modules)
 *******************************************************************************************/

/* The struct SVGContextBmpDec_ is defined further down in this file */
struct SVGContextBmpDec_;

/**
 * Decoding info which is passed to the decoder modules
 */
typedef struct GRL_BMPDEC_dec_info_
{
    SVGImage                    image;              /**< Specified image structure                      */
    struct SVGContextBmpDec_    *p_bmp_ctx;         /**< BMPDEC context for this decoding info          */
    SVGBoolean                  revert;
} GRL_BMPDEC_dec_info;


/**
 * Bitmap decoding context
 */
typedef struct SVGContextBmpDec_ 
{
    SVGInt32                    write_info_cnt;                         /**< Number of queued decoding requests in write queue  */
    SVGInt32                    read_info_cnt;                          /**< Number of queued decoding requests in read queue   */
    struct GRL_BMPDEC_dec_info_ *p_decoding_info[GRL_BMPDEC_QU_NUM];    /**< List of decoding requests                          */
    struct GRL_BMPDEC_dec_info_ *p_write_queue;                         /**< Pointer to queue for writing requests              */
    struct GRL_BMPDEC_dec_info_ *p_read_queue;                          /**< Pointer to queue for reading requests              */
    SVGError                    lastError;                              /**< First error which occurred on this ctx             */
    GRL_lock_id                 busy_lock;                              /**< ID of lock for protecting context                  */
    pthread_t                   dec_tsk_id;                             /**< Task ID of decoding task                           */
    GRL_evtflg_id               dec_flg;                                /**< Flag of decoding task event flag                   */
    SVGBoolean                  queue_running;                          /**< Read queue is being decoded in task                */
    SVGBoolean                  rgb_to_bgr;								/**< internal RGB to RGB conversion should be done      */
} SVGContextBmpDec;


/*******************************************************************************************
 *   API function prototypes
 *******************************************************************************************/

/* [BEGIN SYSCALLS] */


/**
 * Initialize the Bitmap Decoder library and the decoder modules.
 * Has to be called before any other Bitmap Decoder API can be used.
 * 
 * One of the possible values will be set to the global error code:
 *  \li \c SVG_NO_ERROR: Init was successful
 *  \li \c SVG_OUT_OF_MEMORY: Not enough memory for setting up internal structures
 *  \li \c SVG_ALREADY_IN_USE: The Bitmap Decoder was already initialized
 *  \li \c SVG_BMPDEC_NO_DECODER: No decoder module was found during initialization
 *
 * \see svgCloseBmpDec
 */
extern void                svgInitBmpDec( void );


/**
 * Close the Bitmap Decoder.
 * No other Bitmap Decoder API (except svgInitBmpDec)
 * can be called after this function. This function returns
 * its error code directly instead of setting the global error
 * code because the function svgGetBmpDecError is not available 
 * after shutdown.
 *
 * \return      SVG_NO_ERROR            Shutdown was successful
 * \return      SVG_NOT_INITIALIZED     Module was not initialized before
 *
 * \see svgInitBmpDec
 */
extern  SVGError            svgCloseBmpDec( void );


/**
 * Create a new context for the Bitmap Decoder.
 * The context serves as a handle for the subsequent Bitmap Decoder
 * operations. At least one context is required for each task.
 * 
 * One of the possible values will be set to the global error code:
 *  \li \c SVG_NO_ERROR: Init was successful
 *  \li \c SVG_OUT_OF_MEMORY: Not enough memory for setting up internal structures
 *  \li \c SVG_ALREADY_IN_USE: The Bitmap Decoder was already initialized
 *  \li \c SVG_BMPDEC_NO_DECODER: No decoder module was found during initialization
 *
 * \return Context for SVG Bitmap Decoder
 *
 * \see svgDestroyContextBmpDec
 */
extern  SVGContextBmpDec*   svgCreateContextBmpDec( void );


/**
 * Destroy the specified context for the Bitmap Decoder.
 * After calling this function, the context may not be
 * used for decoding anymore.
 *
 * \param[in]   *ctx                    Context for SVG bitmap decoder
 *
 * \see svgCreateContextBmpDec
 */
extern  void                svgDestroyContextBmpDec ( SVGContextBmpDec *ctx );


/**
 * Provide information about an encoded image.
 * The information which can be retrieved depends on the image format.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 * \param[in]   *image                  Pointer to image definition structure. Only the
 *                                      fields type, data_size and datap are required.
 * \param[out]  *image_info             Pointer to image info structure containing the results
 *
 */
extern  void                svgGetImageInfo( SVGContextBmpDec      *ctx,
                                             const SVGImage        *image,
                                             SVGImageInfo          *image_info );


/**
 * List supported decoding formats.
 * As it is possible to deactivate some of the decoder modules by configuration,
 * this function is provided to allow the application to query which formats
 * are currently supported.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 * \param[out]  *formats                Pointer to resulting bitmask which specifies
 *                                      the supported formats ('1' means format is supported)
 */
extern  void                svgGetBmpDecFormats( SVGContextBmpDec       *ctx,
                                                 SVGBmpDecFormats       *formats );


/**
 * Decode the specified image.
 * Target for decoding can either be a SVG Surface or a memory
 * location. The surfaces must either be
 * ARGB8888.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 * \param[in]   *image                  Pointer to image definition structure
 *
 * \see svgFinishBmpDec, svgFlushBmpDec
 */
extern  void                svgDecodeImage( SVGContextBmpDec   *ctx,
                                            const SVGImage     *image );


/**
 * Flush the command list of the specified Bitmap Decoder context.
 * Start the asynchronous execution of the decoding requests which were entered
 * with svgDecodeImage.
 * 
 * \param[in]   *ctx                    Context for Bitmap Decoder
 *
 * \see svgDecodeImage
 */
extern  void                svgFlushBmpDec( SVGContextBmpDec *ctx );


/**
 * Finish the command list of the specified Bitmap Decoder context.
 * Start the synchronous execution of the decoding requests which were entered
 * with svgDecodeImage.
 * 
 * \param       *ctx                    Context for Bitmap Decoder
 *
 * \see svgDecodeImage
 */
extern  void                svgFinishBmpDec( SVGContextBmpDec *ctx );


/**
 * Wait for completion of decoding.
 * If an asynchronous decoding request has been issued with svgFlushBmpDec
 * this function waits until the decoding is finished. Only after this
 * function returns the application can safely assume that all decoded
 * images are ready.
 * 
 * \param       *ctx                    Context for Bitmap Decoder
 *
 * \see svgFlushBmpDec
 */
extern  void                svgWaitBmpDec( SVGContextBmpDec *ctx );


/**
 * Get error code of first error which occurred.
 * The first error which occurred after the last call to
 * this function or context creation is returned. The
 * internal error code is reset afterwards.
 *
 * \param       *ctx                    Context for Bitmap Decoder
 *
 * \return      The error code
 *
 * @see
 */
extern  SVGError            svgGetBmpDecError( SVGContextBmpDec* ctx );

/* [END SYSCALLS] */



/*******************************************************************************************
 *   Internal function types for communication with decoder modules
 *******************************************************************************************/

/**
 * Function type for requesting decoding from Decoder Module
 */
typedef	SVGError (*GRL_BMPDEC_DRAW_IMAGE_FUNCS)( GRL_BMPDEC_dec_info* );


/**
 * Function type for requesting image info from Decoder Module
 */
typedef SVGError (*GRL_BMPDEC_GET_IMAGE_INFO_FUNCS)( const SVGContextBmpDec*, const SVGImage*, SVGImageInfo* );


#ifdef __cplusplus
}
#endif

#endif  /* SVG_BITMAP_DECODER_H */
